
# coding=utf-8
# Copyright 2019 The Google Research Authors.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import torch
import torch.nn as nn
import torch.nn.init as init
import torch.nn.functional as F
# import norms
import numpy as np
from torch.distributions.categorical import Categorical


def conv3x3(in_planes, out_planes, stride=1):
    return nn.Conv2d(in_planes, out_planes, kernel_size=3, stride=stride, padding=1, bias=True)


def conv_init(m):
    classname = m.__class__.__name__
    if classname.find('Conv') != -1:
        init.xavier_uniform(m.weight, gain=np.sqrt(2))
        init.constant(m.bias, 0)
    elif classname.find('BatchNorm') != -1:
        init.constant(m.weight, 1)
        init.constant(m.bias, 0)


class Identity(nn.Module):
    def __init__(self, *args, **kwargs):
        super().__init__()
    def forward(self, x):
        return x


class wide_basic(nn.Module):
    def __init__(self, in_planes, planes, dropout_rate, stride=1, norm=None, leak=.2, i_act_u=0):
        super(wide_basic, self).__init__()
        self.act_u = [nn.LeakyReLU(leak), nn.SiLU(), nn.SELU()][i_act_u]
        self.bn1 = get_norm(in_planes, norm)
        self.conv1 = nn.Conv2d(in_planes, planes, kernel_size=3, padding=1, bias=True)
        self.dropout = Identity() if dropout_rate == 0.0 else nn.Dropout(p=dropout_rate)
        self.bn2 = get_norm(planes, norm)
        self.conv2 = nn.Conv2d(planes, planes, kernel_size=3, stride=stride, padding=1, bias=True)

        self.shortcut = nn.Sequential()
        if stride != 1 or in_planes != planes:
            self.shortcut = nn.Sequential(
                nn.Conv2d(in_planes, planes, kernel_size=1, stride=stride, bias=True),
            )

    def forward(self, x):
        out = self.dropout(self.conv1(self.act_u(self.bn1(x))))
        out = self.conv2(self.act_u(self.bn2(out)))
        out += self.shortcut(x)
        return out


def get_norm(n_filters, norm):
    if norm is None:
        return Identity()
    elif norm == "batch":
        return nn.BatchNorm2d(n_filters, momentum=0.9)
    elif norm == "instance":
        return nn.InstanceNorm2d(n_filters, affine=True)
    elif norm == "layer":
        return nn.GroupNorm(1, n_filters)
    # elif norm == "act":
    #     return norms.ActNorm(n_filters, False)


class JEMWRN(nn.Module):
    """ WRN used at JEM"""

    def __init__(self, depth, widen_factor, num_classes=10, in_channel=3,
                 sum_pool=False, norm=None, in_norm=None, leak=.2, dropout_rate=0.0,
                 debug=False, i_act_u=0, dataset='cifar10'):
        super(JEMWRN, self).__init__()
        self.leak = leak
        self.in_planes = 16
        self.sum_pool = sum_pool
        self.norm = norm
        self.in_norm = in_norm if in_norm is not None else (lambda x: x)
        self.act_u = [nn.LeakyReLU(leak), nn.SiLU(), nn.SELU()][i_act_u]

        assert ((depth-4)%6 ==0), 'Wide-resnet depth should be 6n+4'
        n = (depth-4)//6
        k = widen_factor

        print('| Wide-Resnet %dx%d' %(depth, k))
        if debug :
            nStages = [2, 2 * k, 3 * k, 4 * k]
        else:
            nStages = [16, 16*k, 32*k, 64*k]

        self.conv1 = conv3x3(in_channel, nStages[0])
        self.layer1 = self._wide_layer(wide_basic, nStages[1], n, dropout_rate, stride=1, i_act_u=i_act_u)
        self.layer2 = self._wide_layer(wide_basic, nStages[2], n, dropout_rate, stride=2, i_act_u=i_act_u)
        self.layer3 = self._wide_layer(wide_basic, nStages[3], n, dropout_rate, stride=2, i_act_u=i_act_u)
        self.bn1 = get_norm(nStages[3], self.norm)

        self.size = 32
        if dataset in ['imagenet', 'img256', 'img128', 'img64', 'tinyimagenet',
                       'celeba128', 'celeba256', 'stl10']:
            self.size = 64
        if self.size >= 64:
            self.layer4 = self._wide_layer(wide_basic, nStages[4], n, dropout_rate, stride=2, i_act_u=i_act_u)
            self.bn1 = get_norm(nStages[4], self.norm)
        i = 4 if self.size >= 64 else 3
        self.last_dim = nStages[i]


    def _wide_layer(self, block, planes, num_blocks, dropout_rate, stride, i_act_u):
        strides = [stride] + [1]*(num_blocks-1)
        layers = []

        for stride in strides:
            layers.append(block(self.in_planes, planes, dropout_rate, stride, norm=self.norm, i_act_u=i_act_u))
            self.in_planes = planes

        return nn.Sequential(*layers)

    # @profile
    def forward(self, x):
        x1 = self.in_norm(x);
        out1 = self.conv1(x1)
        out2 = self.layer1(out1)
        out3 = self.layer2(out2)
        out4 = self.layer3(out3)
        if x.shape[-1] >= 64:
            out4 = self.layer4(out4)
        out5 = self.act_u(self.bn1(out4))
        if self.sum_pool:
            out6 = out5.view(out5.size(0), out5.size(1), -1).sum(2)
        else:
            out6 = F.avg_pool2d(out5, 8)
        out7 = out6.view(out6.size(0), -1)
        return out7
